<?php

namespace App\Http\Controllers;

use App\Http\Requests\UserRequest;
use App\Models\Pessoa;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class UserController extends Controller
{
    // Mostra a view da listagem dos utilizadores
    public function index(){
        
        // Recupera todos os utilizadores com seus dados pessoais
        $users = User::orderBy('id', 'desc')->with(['pessoa'])->paginate(6);
        $ultimoregisto = User::orderBy('id', 'desc')->first();

        return view('Administracao.Users', compact('users', 'ultimoregisto'));
    }

    public function store(UserRequest $request){

        // Validar formulario de cadastro do utilizador
        $request->validated();

        // Iniciando uma transação para garantir a integridade dos dados
        DB::beginTransaction();

        try {
            // Cadastrando a pessoa no banco de dados
            $pessoa = new Pessoa;
            $pessoa->name = $request->name;
            $pessoa->nID = $request->nID;
            $pessoa->tipoID = $request->tipoID;
            $pessoa->genero = $request->genero;
            $pessoa->dataNasc = $request->dataNasc;
            $pessoa->tipoConta = $request->tipoConta;
            $pessoa->contactoPrimario = $request->contactoPrimario;
            $pessoa->status = 'on';
            $pessoa->email = $request->email;
            $pessoa->telefone = $request->telefone;
            $pessoa->obs = $request->obs ?? '';
            $pessoa->save();

            // Cadastrando o utilizador no banco de dados
            $user = new User;
            $user->name = $request->name;
            $user->email = $request->email;
            $user->password = Hash::make($request->password);
            $user->admin = $request->has('admin') ? true : false;
            $user->client = $request->has('client') ? true : false;
            $user->id_pessoa = $pessoa->id;
            $user->save();

            // Confirma a transação
            DB::commit();

            // Recupera o último registro para possível uso na impressão
            $ultimoregisto = User::orderBy('id', 'desc')->first();

            return redirect()->route('users')->with('msg', 'Utilizador cadastrado com sucesso!');

        } catch (\Exception $e) {
            // Reverte a transação em caso de erro
            DB::rollback();
            return redirect()->route('users')->with('error', 'Erro ao cadastrar utilizador: ' . $e->getMessage());
        }
    }

    public function update(UserRequest $request, $id) 
    {
        // Validar formulário de atualização do utilizador
        $request->validated();

        // Iniciando uma transação para garantir a integridade dos dados
        DB::beginTransaction();

        try {
            // Busca o utilizador
            $user = User::findOrFail($id);
            
            // Busca a pessoa associada
            $pessoa = Pessoa::findOrFail($user->id_pessoa);
            
            // Atualiza os dados da pessoa
            $pessoa->name = $request->name;
            $pessoa->nID = $request->nID;
            $pessoa->tipoID = $request->tipoID;
            $pessoa->genero = $request->genero;
            $pessoa->dataNasc = $request->dataNasc;
            $pessoa->tipoConta = $request->tipoConta;
            $pessoa->contactoPrimario = $request->contactoPrimario;
            $pessoa->status = $request->status ?? 'on';
            $pessoa->email = $request->email;
            $pessoa->telefone = $request->telefone;
            $pessoa->obs = $request->obs ?? '';
            $pessoa->save();
            
            // Atualiza os dados do utilizador
            $user->name = $request->name;
            $user->email = $request->email;
            
            // Atualiza a senha apenas se uma nova senha for fornecida
            if (!empty($request->password)) {
                $user->password = Hash::make($request->password);
            }
            
            $user->admin = $request->has('admin') ? true : false;
            $user->client = $request->has('client') ? true : false;
            $user->save();
            
            // Confirma a transação
            DB::commit();
            
            return redirect()->route('users')->with('msg', 'Utilizador atualizado com sucesso!');
            
        } catch (\Exception $e) {
            // Reverte a transação em caso de erro
            DB::rollback();
            return redirect()->route('users')->with('error', 'Erro ao atualizar utilizador: ' . $e->getMessage());
        }
    }

    public function show($id)
    {
        $user = User::with('pessoa')->findOrFail($id);
        return response()->json($user);
    }
    
    public function destroy($id)
    {
        try {
            $user = User::findOrFail($id);
            $user->delete(); // A cascade delete will also remove the pessoa record due to the foreign key constraint
            
            return redirect()->route('users')->with('msg', 'Utilizador removido com sucesso!');
        } catch (\Exception $e) {
            return redirect()->route('users')->with('error', 'Erro ao remover utilizador: ' . $e->getMessage());
        }
    }
}
