<?php

namespace App\Http\Controllers;

use App\Models\Banco;
use App\Models\ComprovativoPagamento;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class ComprovativoPagamentoController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $comprovativos = ComprovativoPagamento::with(['bancoOrigem', 'bancoDestino', 'user', 'validador'])
            ->orderBy('id', 'desc')
            ->paginate(10);

        $bancos = Banco::all();
        $users = User::all();

        return view('Administracao.ComprovativosPagamento', compact('comprovativos', 'bancos', 'users'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Validação dos campos
        $request->validate([
            'arquivo_comprovativo' => 'required|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'data_hora' => 'required|date',
            'banco_origem_id' => 'required|exists:bancos,id',
            'banco_destino_id' => 'required|exists:bancos,id',
            'users_id' => 'required|exists:users,id',
            'observacao' => 'nullable|string',
        ]);

        // Processamento do arquivo de comprovativo
        if ($request->hasFile('arquivo_comprovativo') && $request->file('arquivo_comprovativo')->isValid()) {
            $requestFile = $request->arquivo_comprovativo;
            $extension = $requestFile->extension();

            $fileName = md5($requestFile->getClientOriginalName().strtotime('now')).'.'.$extension;
            $requestFile->move(public_path('Doc/ComprovativosPagamento'), $fileName);

            // Criar o comprovativo de pagamento
            $comprovativo = new ComprovativoPagamento;
            $comprovativo->arquivo_comprovativo = $fileName;
            $comprovativo->data_hora = $request->data_hora;
            $comprovativo->banco_origem_id = $request->banco_origem_id;
            $comprovativo->banco_destino_id = $request->banco_destino_id;
            $comprovativo->users_id = $request->users_id;
            $comprovativo->observacao = $request->observacao;
            $comprovativo->status = 'pendente';

            $comprovativo->save();

            return redirect()->route('comprovativos.pagamento')->with('msg', 'Comprovativo de Pagamento registado com sucesso!');
        }

        return redirect()->back()->with('error', 'Erro ao carregar o arquivo do comprovativo.');
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $comprovativo = ComprovativoPagamento::with(['bancoOrigem', 'bancoDestino', 'user', 'validador'])
            ->findOrFail($id);

        return response()->json($comprovativo);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        // Validação dos campos
        $request->validate([
            'arquivo_comprovativo' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'data_hora' => 'required|date',
            'banco_origem_id' => 'required|exists:bancos,id',
            'banco_destino_id' => 'required|exists:bancos,id',
            'users_id' => 'required|exists:users,id',
            'observacao' => 'nullable|string',
        ]);

        // Buscar o comprovativo
        $comprovativo = ComprovativoPagamento::findOrFail($id);

        // Atualizar os campos
        $comprovativo->data_hora = $request->data_hora;
        $comprovativo->banco_origem_id = $request->banco_origem_id;
        $comprovativo->banco_destino_id = $request->banco_destino_id;
        $comprovativo->users_id = $request->users_id;
        $comprovativo->observacao = $request->observacao;

        // Processar arquivo de comprovativo (se enviado)
        if ($request->hasFile('arquivo_comprovativo') && $request->file('arquivo_comprovativo')->isValid()) {
            $requestFile = $request->arquivo_comprovativo;
            $extension = $requestFile->extension();

            $fileName = md5($requestFile->getClientOriginalName().strtotime('now')).'.'.$extension;
            $requestFile->move(public_path('Doc/ComprovativosPagamento'), $fileName);

            // Apagar arquivo antigo se existir
            if (! empty($comprovativo->arquivo_comprovativo)) {
                $caminhoArquivoAntigo = public_path('Doc/ComprovativosPagamento/'.$comprovativo->arquivo_comprovativo);
                if (file_exists($caminhoArquivoAntigo)) {
                    unlink($caminhoArquivoAntigo);
                }
            }

            $comprovativo->arquivo_comprovativo = $fileName;
        }

        // Salvar as alterações
        $comprovativo->save();

        return redirect()->route('comprovativos.pagamento')->with('msg', 'Comprovativo de Pagamento atualizado com sucesso!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $comprovativo = ComprovativoPagamento::findOrFail($id);

        // Apagar o arquivo do comprovativo
        if (! empty($comprovativo->arquivo_comprovativo)) {
            $caminhoArquivo = public_path('Doc/ComprovativosPagamento/'.$comprovativo->arquivo_comprovativo);
            if (file_exists($caminhoArquivo)) {
                unlink($caminhoArquivo);
            }
        }

        $comprovativo->delete();

        return redirect()->route('comprovativos.pagamento')->with('msg', 'Comprovativo de Pagamento removido com sucesso!');
    }

    /**
     * Validar ou invalidar um comprovativo de pagamento.
     */
    public function validar(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:validado,invalidado',
            'observacao' => 'nullable|string',
        ]);

        $comprovativo = ComprovativoPagamento::findOrFail($id);
        $comprovativo->status = $request->status;

        if ($request->filled('observacao')) {
            $comprovativo->observacao = $request->observacao;
        }

        $comprovativo->validado_por = Auth::id();
        $comprovativo->data_validacao = Carbon::now();
        $comprovativo->save();

        $statusMsg = $request->status == 'validado' ? 'validado' : 'invalidado';

        return redirect()->route('comprovativos.pagamento')->with('msg', "Comprovativo de Pagamento {$statusMsg} com sucesso!");
    }

    /**
     * Busca parametrizada de comprovativos de pagamento.
     */
    public function search(Request $request)
    {
        $query = ComprovativoPagamento::with(['bancoOrigem', 'bancoDestino', 'user', 'validador']);

        // Filtro por data inicial
        if ($request->filled('start_date')) {
            $query->whereDate('data_hora', '>=', $request->start_date);
        }

        // Filtro por data final
        if ($request->filled('end_date')) {
            $query->whereDate('data_hora', '<=', $request->end_date);
        }

        // Filtro por banco de origem
        if ($request->filled('banco_origem')) {
            $query->where('banco_origem_id', $request->banco_origem);
        }

        // Filtro por banco de destino
        if ($request->filled('banco_destino')) {
            $query->where('banco_destino_id', $request->banco_destino);
        }

        // Filtro por funcionário
        if ($request->filled('funcionario')) {
            $query->where('users_id', $request->funcionario);
        }

        // Filtro por status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Executar a consulta com paginação
        $comprovativos = $query->orderBy('id', 'desc')->paginate(10)->withQueryString();
        $bancos = Banco::all();
        $users = User::all();

        return view('Administracao.ComprovativosPagamento', compact('comprovativos', 'bancos', 'users'));
    }
}
